/*  -*- pse-c -*-
 *----------------------------------------------------------------------------
 * Filename: iegd_interface.c
 * $Revision: 1.18 $
 *----------------------------------------------------------------------------
 * Gart and DRM driver for Intel Embedded Graphics Driver
 * Copyright  2008, Intel Corporation.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms and conditions of the GNU General Public License,
 * version 2, as published by the Free Software Foundation.
 *
 * This program is distributed in the hope it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
 * more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 
 * 51 Franklin St - Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */

/* Copyright 2003 - 2005 Intel Corporation. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *	Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *
 *      Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *
 *      Neither the name Intel Corporation nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 */

#include "iegd.h"
#include "drmP.h"
#include "drm.h"

#include "iegd_drm.h"
#include "iegd_drv.h"
#include "psb_intregs.h"

#ifndef MSR_IA32_CR_PAT
#define MSR_IA32_CR_PAT 0x0277
#endif
#ifndef _PAGE_PAT
#define _PAGE_PAT 0x080
#endif
extern void agp_init_pat(void);
extern int agp_use_pat (void);

int drm_irq_install(drm_device_t *dev);

/* get intel_buffer_fops from the interface_###.c files */
extern struct file_operations intel_buffer_fops;

#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,5)
extern struct vm_operations_struct iegd_plb_vm_ops;
#endif


/* Global variable to keep track the amount of memory we are using */
static int memory;

/* Our own mmap function to memory map physical to user space memory
 */
int intel_mmap_buffers(struct file *filp, struct vm_area_struct *vma)
{
	DRM_DEBUG("\n");

	lock_kernel();
	vma->vm_flags |= (VM_IO | VM_RESERVED);
	vma->vm_file = filp;
	unlock_kernel();
	
	DRM_DEBUG("VM_OFFSET(vma):%#x\n",(unsigned int)VM_OFFSET(vma));
	if (REMAP_PAGE( vma,
			vma->vm_start,
			VM_OFFSET(vma),
			(vma->vm_end - vma->vm_start),
			pgprot_noncached(vma->vm_page_prot))){
	 return -EAGAIN;
	}

	return 0;
}

/* IOCTL to Allocate size pages and mmap it to the client calling it with
 * corresponding virtual address
 */
int intel_getpages( drm_device_t *dev, struct file *filp, unsigned long arg ){

	drm_intel_getpages_t getpages;
	/* allocate some bytes */
	unsigned long bytes;
	int order;
	int size;

	unsigned long address;
	unsigned long phy_address;
	unsigned long offset;

	struct page *pg;

	unsigned long virtual;
	struct file_operations *old_fops;

	intel_device_private_t *dev_ptr=dev->dev_private;
	drm_intel_listpages_t *page;
	drm_intel_list_t     *list;

	DRM_DEBUG("\n");
	/* copy user arguments */
	if(copy_from_user(&getpages, (void __user *) arg, sizeof(getpages))){
		return -EFAULT;
	}

	bytes=getpages.size;
	/* Check to see if this allocation would exceed 16MEG in total memory
	 * This is to prevent denial of service attack. 16Meg should be enough.
	 */
	if((memory+bytes)>MB(16) ){
		/* We exceeded 16MEG. Bail out */
		DRM_ERROR("Total memory allocated exceeded 16Meg!\n");
		return -EFAULT;
	}

	/*number of pages that are needed*/
	size=bytes>>PAGE_SHIFT;
	if(bytes & ~(PAGE_SIZE*size)){
		++size;
	}
	order=ORDER(size);
	DRM_DEBUG("Allocating bytes:%#lx,size:%d,order:%d\n",
		(unsigned long)bytes,size,order); 
	/* allocate the pages */
	/* returns kernel logical address. 
	 * Is this the same as the kernel virtual address?? 
	 */
	address=ALLOC_PAGES(order,0);
	if(!address){
		DRM_ERROR("Can't get pages\n");
		return -EFAULT;
	}
	phy_address=__pa(address);

	/* Find virtual address of the phys address */
	pg=virt_to_page((void *)address);
	offset=pg->index;
	/* Find the number of bytes that is actually allocated */
	size=PAGE_SIZE<<order;
	DRM_DEBUG("Allocated address:%#lx,page offset:%#lx,phy_address:%#lx\n",
		address,offset,phy_address); 

	/*do_mmap on the logical address and return virtual address */
	down_write(&current->mm->mmap_sem);

	old_fops= (struct file_operations *)filp->f_op;
	filp->f_op=&intel_buffer_fops;

	virtual=do_mmap(filp,0,size,PROT_READ|PROT_WRITE,MAP_SHARED,phy_address);

	filp->f_op=old_fops;
	up_write(&current->mm->mmap_sem);
	DRM_DEBUG("Mmaped virtual:%#lx,address:%#lx\n",virtual,
		(unsigned long)__va(phy_address));
	if(virtual > -1024UL){
		DRM_ERROR("mmap failed:%d\n",(int)virtual);
		return -EFAULT;
	}
	getpages.phy_address=phy_address;
	getpages.virt_address=virtual;
	getpages.size=size;
	getpages.offset=offset;

	DRM_DEBUG("Mmap success requested size:%d (%d)\n",
		getpages.size,(int)bytes);

	/* alloc the page to be put into the linked list */
	page=ALLOC(sizeof(*page),DRM_MEM_DRIVER);
	if(!page){
		DRM_DEBUG("Can't alloc list for page\n");
		return -ENOMEM;
	}

	/*page->pid=current->pid;*/
	page->pid=current->group_leader->pid;
	page->size=size;
	page->phy_address=phy_address;
	page->virt_address=virtual;
	page->offset=offset;

	DRM_DEBUG("parent pid:%d,pid:%d,group_leader->pid:%d\n"
		,current->parent->pid,current->pid,current->group_leader->pid);
	/* Alloc the list to be added then add it to the linked list */
	list=ALLOC(sizeof(*list),DRM_MEM_DRIVER);
	if(!list){
		DRM_DEBUG("Can't alloc list for page\n");
		FREE(page,sizeof(*page),0);
		return -ENOMEM;
	}
	memset(list,0,sizeof(*list));
	list->page=page;
	LOCK_DRM(dev);
	list_add(&list->head,&dev_ptr->pagelist->head);
	UNLOCK_DRM(dev);
	if(copy_to_user((void __user *) arg,&getpages,sizeof(getpages))){
		return -EFAULT;
	}
	/* update the total amount of memory we use */
	memory+=size;
	DRM_DEBUG("memory has:%d bytes\n",memory);

return 0;
}

/* IOCTL to free pages that are allocated by getpages 
 */
int intel_freepages( drm_device_t *dev, unsigned long arg ){

	drm_intel_freepages_t freepages;
	/* allocate some bytes */
	unsigned long bytes;
	int order;
	int size;

	intel_device_private_t *dev_ptr=dev->dev_private;
	drm_intel_listpages_t *page;
	drm_intel_list_t *r_list=NULL;
	struct list_head *pagelist;

	DRM_DEBUG("Freeing pages\n");
	/* copy user arguments */
	if(copy_from_user(&freepages, (void __user *) arg, sizeof(freepages))){
		return -EFAULT;
	}

	bytes=freepages.size;
	/*number of pages that are needed*/
	size=bytes>>PAGE_SHIFT;
	if(bytes & ~(PAGE_SIZE*size)){
		++size;
	}
	order=ORDER(size);
	DRM_DEBUG("bytes:%d,size:%d,order:%d,phy_address:%#lx\n",(int)bytes,(int)size,(int)order,freepages.phy_address);

	/* free the pages */
	DRM_DEBUG("freeing address:%#lx,size:%#lx\n",(unsigned long)__va(freepages.phy_address),(unsigned long)bytes);

	DRM_DEBUG("parent pid:%d,pid:%d,group_leader->pid:%d\n"
		,current->parent->pid,current->pid,current->group_leader->pid);
	/* See if the requested address is in our page list */
	LOCK_DRM(dev);
	pagelist=&dev_ptr->pagelist->head;
	list_for_each(pagelist,&dev_ptr->pagelist->head){
		r_list=list_entry(pagelist,drm_intel_list_t,head);
		if((r_list->page->pid==current->group_leader->pid) 
			&& (r_list->page->phy_address==freepages.phy_address)){
		
		DRM_DEBUG("found pid:%d\n",current->group_leader->pid);
		DRM_DEBUG("size:%d\n",r_list->page->size);
		DRM_DEBUG("phy_address:%#lx\n",r_list->page->phy_address);
		DRM_DEBUG("virt_add:%#lx\n",r_list->page->virt_address);
		DRM_DEBUG("offset:%#lx\n",r_list->page->offset);
		
		break;
		}

	}
	if(pagelist==(&dev_ptr->pagelist->head)){
		DRM_DEBUG("Can't find pages alloc for pid:%d\n",current->pid);
		UNLOCK_DRM(dev);
		return -EINVAL;
	}

	/* munmap the region 1st */
	down_write(&current->mm->mmap_sem);
	DRM_DEBUG("Unmapping virt_address:%#lx\n",freepages.virt_address);
	do_munmap(current->mm,freepages.virt_address,bytes);
	up_write(&current->mm->mmap_sem);

	/* Free the pages! */
	FREE_PAGES((unsigned long)__va(freepages.phy_address),order,0);

	/* Free the page list */
	page=r_list->page;
	list_del(pagelist);
	size=r_list->page->size;
	FREE(pagelist,sizeof(*pagelist),0);
	FREE(page,sizeof(*page),0);
	UNLOCK_DRM(dev);

	/* update the total memory that we use */
	memory-=size;
	DRM_DEBUG("memory has:%d bytes\n",memory);
	return 0;
}
/* This code is copied verbatim from the DRM module code in 
 * IKM/val/drm/drmv11p0/drm_irq.c.  It's here because we
 * need to activate interrupt handling, but for some reason the DRM module
 * only exports the routine to disable interrupt handling drm_irq_uninstall(),
 * and not the one to install.
 * 
 * This could be problematic when new DRM versions appear.
 * 
 * Fortunately, should a new DRM version appear, it should export
 * drm_irq_install(), and then this source won't be needed at all; the
 * code should compile cleanly with an external reference if this
 * static version is removed completely.
 */
int drm_irq_install(drm_device_t * dev)
{
	int ret;
	unsigned long sh_flags = 0;

	if (!drm_core_check_feature(dev, DRIVER_HAVE_IRQ))
		return -EINVAL;

	if (dev->irq == 0)
		return -EINVAL;

	mutex_lock(&dev->struct_mutex);

	/* Driver must have been initialized */
	if (!dev->dev_private) {
		mutex_unlock(&dev->struct_mutex);
		return -EINVAL;
	}

	if (dev->irq_enabled) {
		mutex_unlock(&dev->struct_mutex);
		return -EBUSY;
	}
	dev->irq_enabled = 1;
	mutex_unlock(&dev->struct_mutex);

	DRM_DEBUG("%s: irq=%d\n", __FUNCTION__, dev->irq);

	/*
	if (drm_core_check_feature(dev, DRIVER_IRQ_VBL)) {
		init_waitqueue_head(&dev->vbl_queue);

		spin_lock_init(&dev->vbl_lock);

		INIT_LIST_HEAD(&dev->vbl_sigs.head);
		INIT_LIST_HEAD(&dev->vbl_sigs2.head);

		dev->vbl_pending = 0;
	}
	*/

	/* Before installing handler */
	dev->driver->irq_preinstall(dev);

	/* Install handler */
	if (drm_core_check_feature(dev, DRIVER_IRQ_SHARED))
		sh_flags = IRQF_SHARED;

	ret = request_irq(dev->irq, dev->driver->irq_handler,
			  sh_flags, dev->devname, dev);
	if (ret < 0) {
		mutex_lock(&dev->struct_mutex);
		dev->irq_enabled = 0;
		mutex_unlock(&dev->struct_mutex);
		return ret;
	}

	/* After installing handler */
	dev->driver->irq_postinstall(dev);

	return 0;
}

/* IOCTL to init the info that is needed by the client 
 */
int intel_drm_info_init( drm_device_t *dev, unsigned long arg ){

	intel_drm_info_t info;
	intel_drm_info_t *info_ptr;
	intel_device_private_t *dev_ptr;

	DRM_DEBUG("info init succesful dev_private:%#lx\n",(unsigned long)dev->dev_private);
	dev_ptr=dev->dev_private;
	/* See if dev_private is already allocated */
	if(!dev->dev_private){
		DRM_ERROR("dev_private not allocated!\n");
		return 0;
	}
	info_ptr=dev_ptr->info_ptr;
	/* See if info is already allocated */
	if(info_ptr->device_id){
		DRM_ERROR("Info already allocated!\n");
		return 0;
	}

	/* copy user arguments */
	if(copy_from_user(&info, (void __user *) arg, sizeof(info))){
		return -EFAULT;
	}

	info_ptr->device_id=info.device_id;
	info_ptr->revision=info.revision;
	info_ptr->video_memory_offset=info.video_memory_offset;
	info_ptr->video_memory_size=info.video_memory_size;
	info_ptr->hw_status_offset=info.hw_status_offset;
	DRM_DEBUG("device_id:%#lx,revision:%#lx,offset:%#lx,size:%#lx,hw_status_offset:%lx\n",
		info_ptr->device_id,info_ptr->revision,
		info_ptr->video_memory_offset,info_ptr->video_memory_size,
		info_ptr->hw_status_offset);
return 0;
}
/* IOCTL to get the info that is needed by the client 
 */
int intel_drm_info_get( drm_device_t *dev, unsigned long arg ){

	intel_drm_info_t info;
	intel_device_private_t *dev_ptr=dev->dev_private;
	intel_drm_info_t *info_ptr=dev_ptr->info_ptr;

	DRM_DEBUG("Info get device_id:%#lx,revision:%#lx,offset:%#lx,size:%#lx, hw_status_offset:%lx\n",
		info_ptr->device_id,info_ptr->revision,
		info_ptr->video_memory_offset,info_ptr->video_memory_size,
		info_ptr->hw_status_offset);

	info.device_id=info_ptr->device_id;
	info.revision=info_ptr->revision;
	info.video_memory_offset=info_ptr->video_memory_offset;
	info.video_memory_size=info_ptr->video_memory_size;
	info.hw_status_offset=info_ptr->hw_status_offset;

	if(copy_to_user((void __user *) arg,&info,sizeof(info))){
		return -EFAULT;
	}

return 0;
}

/* initialise structure for link list and driver info in dev_private */
int intel_postinit(intel_device_private_t **priv){

	intel_drm_info_t *info_ptr;
	intel_device_private_t *dev_ptr;
	DRM_DEBUG("\n");
	/* allocate info to be stored */
	dev_ptr=ALLOC(sizeof(intel_device_private_t),DRM_MEM_DRIVER);

	if(!dev_ptr){
		return -ENOMEM;
	}

	DRM_DEBUG("dev_ptr allocation succesful\n");

	memset(dev_ptr,0,sizeof(intel_device_private_t));	
	*priv=dev_ptr;

	info_ptr=ALLOC(sizeof(intel_drm_info_t),DRM_MEM_DRIVER);

	if(!info_ptr){
		return -ENOMEM;
	}

	DRM_DEBUG("Info_ptr allocation succesful\n");
	memset(info_ptr,0,sizeof(intel_drm_info_t));
	dev_ptr->info_ptr=info_ptr;

	dev_ptr->pagelist=ALLOC(sizeof(*dev_ptr->pagelist),DRM_MEM_DRIVER);

	if(!dev_ptr->pagelist){
		return -ENOMEM;
	}

	DRM_DEBUG("pagelist allocation succesful\n");
	memset(dev_ptr->pagelist,0,sizeof(*dev_ptr->pagelist));
	INIT_LIST_HEAD(&dev_ptr->pagelist->head);
	/* Initialise global variable to zero when we start up */
	memory=0;
	DRM_DEBUG("Initialised memory:%d\n",memory);

return 0;

}
/* check and free pages of client that is closing the fd */
int intel_prerelease(drm_device_t *dev){
	unsigned long bytes;
	int order;
	int size;

	intel_device_private_t *dev_ptr=dev->dev_private;
	drm_intel_listpages_t *page;
	drm_intel_list_t *r_list=NULL;
	struct list_head *pagelist, *pagelist_next;

	DRM_DEBUG("Client closing freeing pages alloc to it\n");


	/* Search for the page list has been added and free it */

	LOCK_DRM(dev);

	/* The changes to this function are copied form 8.1 */
	/* I've no idea why, but sometimes during bootup the dev_private
 	* field can show up as NULL.  Guarding against this for now...
 	*/	
	if (dev_ptr != NULL) {

	pagelist=&dev_ptr->pagelist->head;
	list_for_each_safe(pagelist,pagelist_next,&dev_ptr->pagelist->head){
		r_list=list_entry(pagelist,drm_intel_list_t,head);
		if(r_list->page->pid==current->group_leader->pid){
#if 0			
			printk("found pid:%d\n",current->pid);
			printk("size:%d\n",r_list->page->size);
			printk("phy_address:%#lx\n",r_list->page->phy_address);
			printk("virt_add:%#lx\n",r_list->page->virt_address);
			printk("offset:%#lx\n",r_list->page->offset);
#endif		
			bytes=r_list->page->size;

			/*number of pages that are needed*/

			size=bytes>>PAGE_SHIFT;
			if(bytes & ~(PAGE_SIZE*size)){
				++size;
			}
			order=ORDER(size);
			/* free the pages */

#if 0			
			printk("freeing address:%#lx,size:%#lx\n"
				,(unsigned long)__va(r_list->page->phy_address)
				,(unsigned long)bytes);
#endif		

			FREE_PAGES((unsigned long)__va(r_list->page->phy_address)
				,order,0);

			/* remove from list and free the resource */

			page=r_list->page;
			list_del(pagelist);
			FREE(pagelist,sizeof(*pagelist),0);
			FREE(page,sizeof(*page),0);
			/* update the total memory that we use */
			memory-=bytes;
			DRM_DEBUG("memory:%d bytes\n",memory);
		}
 		}	 
	}

	UNLOCK_DRM(dev);

	return 0;

}

int drm_plb_mmap(struct file *filp, struct vm_area_struct *vma)
{
#if LINUX_VERSION_CODE > KERNEL_VERSION(2,6,5)
	drm_file_t *priv = filp->private_data;
	drm_device_t *dev = priv->head->dev;
	drm_map_t *map = NULL;
	drm_map_list_t *r_list;
	unsigned long offset = 0;
	struct list_head *list;
	drm_vma_entry_t *vma_entry;

	DRM_DEBUG("drm_plb_mmap:  start = 0x%lx, end = 0x%lx, offset = 0x%lx\n",
		vma->vm_start, vma->vm_end, VM_OFFSET(vma));

	if (!priv->authenticated) {
		DRM_DEBUG("Did not authenticate");
		return -EACCES;
	} else {
		DRM_DEBUG("Authenticate successful");
	}

	/* A sequential search of a linked list is
	 * fine here because: 1) there will only be
	 * about 5-10 entries in the list and, 2) a
	 * DRI client only has to do this mapping
	 * once, so it doesn't have to be optimized
	 * for performance, even if the list was a
	 * bit longer. */

	/* FIXME: Temporary fix. */
	LIST_FOR_EACH(list, dev) {

		r_list = list_entry(list, drm_map_list_t, head);
		map = r_list->map;
		if (!map)
			continue;
		if (r_list->user_token == VM_OFFSET(vma))
			break;
	}

	if (!map || ((map->flags & _DRM_RESTRICTED) && !capable(CAP_SYS_ADMIN)))
		return -EPERM;

	/* Check for valid size. */
	if (map->size != vma->vm_end - vma->vm_start) {
		return -EINVAL;
	}

	if (!capable(CAP_SYS_ADMIN) && (map->flags & _DRM_READ_ONLY)) {
		vma->vm_flags &= ~(VM_WRITE | VM_MAYWRITE);
		pgprot_val(vma->vm_page_prot) &= ~_PAGE_RW;
	}

	switch (map->type) {

	case _DRM_AGP:
	case _DRM_FRAME_BUFFER:
		if (agp_use_pat()) {
			pgprot_val(vma->vm_page_prot) &= ~(_PAGE_PWT | _PAGE_PCD);
			pgprot_val(vma->vm_page_prot) |= _PAGE_PAT;
			vma->vm_flags |= VM_IO;	/* not in core dump */

			offset = VM_OFFSET(vma) - agp_bridge->gart_bus_addr;
			vma->vm_ops = &iegd_plb_vm_ops;
			break;
		}

		/* Fallthrough */
	case _DRM_REGISTERS:
		if (boot_cpu_data.x86 > 3 && map->type != _DRM_AGP) {
			pgprot_val(vma->vm_page_prot) |= _PAGE_PCD;
			pgprot_val(vma->vm_page_prot) &= ~_PAGE_PWT;
		}
		vma->vm_flags |= VM_IO;	/* not in core dump */
		offset = VM_OFFSET(vma) - agp_bridge->gart_bus_addr;

		vma->vm_ops = &iegd_plb_vm_ops;
		break;
	case _DRM_SHM:
	case _DRM_CONSISTENT:
	case _DRM_SCATTER_GATHER:
		DRM_DEBUG("Fall through to original mmap\n");
		return drm_mmap(filp, vma);
		break;
	default:
		return -EINVAL;	/* This should never happen. */
	}


	vma->vm_flags |= VM_RESERVED;	/* Don't swap */

	vma->vm_file = filp;	/* Needed for drm_vm_open() */

	vma_entry = drm_alloc(sizeof(*vma_entry), DRM_MEM_VMAS);
	if (vma_entry) {
		/*
		 * FIXME: Temporary fix. Will figure out later
		 */
		INSERT_VMA();
	}

#endif
	return 0;
}

int psb_init(intel_device_private_t *priv)
{
	DRM_INIT_WAITQUEUE(&priv->event_queue);
	spin_lock_init(&priv->irqmask_lock);
	priv->event_present = 0;
	priv->out_vdc = 0;
	priv->out_sgx = 0;
	priv->out_sgx2 = 0;
	priv->out_mtx = 0;

	return 0;
}

int intel_drm_plb_interrupts( drm_device_t *dev, void *data )
{
	intel_device_private_t *priv;
	interrupt_info_t plb_info;
	unsigned long irqflags;
	int ret = 0;
	priv=(intel_device_private_t *)dev->dev_private;

	if(copy_from_user(&plb_info, (void __user *) data, sizeof(plb_info))) {
		return -EFAULT;
	}

	/* USW15 definition of in and out
	 *
	 * in/out[0] VDC
	 * in/out[1] sgx
	 * in/out[2] sgx2
	 * in/out[3] msvdx
	 */

	plb_info.out[0]=0;
	plb_info.out[1]=0;
	plb_info.out[2]=0;
	plb_info.out[3]=0;

	switch (plb_info.req_type) {
	case CLEAR_INT:

		plb_info.in[0] &= priv->vdc_irq_mask;
		plb_info.in[1] &= priv->sgx_irq_mask;
		plb_info.in[2] &= priv->sgx_irq_mask2;
		plb_info.in[3] &= priv->msvdx_irq_mask;

		if (plb_info.in[0] || plb_info.in[1] ||
			plb_info.in[2] || plb_info.in[3]) {

			spin_lock_irqsave(&priv->irqmask_lock, irqflags);
			priv->out_vdc &= ~plb_info.in[0];
			plb_info.out[0] = priv->out_vdc;

			priv->out_sgx &= ~plb_info.in[1];
			plb_info.out[1] = priv->out_sgx;

			priv->out_sgx2 &= ~plb_info.in[2];
			plb_info.out[2] = priv->out_sgx2;

			priv->out_mtx &= ~plb_info.in[3];
			plb_info.out[3] = priv->out_mtx;
			spin_unlock_irqrestore(&priv->irqmask_lock, irqflags);

			plb_info.req_status = INT_CLEARED;

		} else {
			plb_info.req_status = INT_NOOP;
		}

		break;

	case READ_INT:


		plb_info.out[0] = priv->out_vdc;
		plb_info.out[1] = priv->out_sgx;
		plb_info.out[2] = priv->out_sgx2;
		plb_info.out[3] = priv->out_mtx;
		plb_info.req_status = INT_READ;

		break;

	case WAIT_INT:

		plb_info.in[0] &= priv->vdc_irq_mask;
		plb_info.in[1] &= priv->sgx_irq_mask;
		plb_info.in[2] &= priv->sgx_irq_mask2;
		plb_info.in[3] &= priv->msvdx_irq_mask;

		if (plb_info.in[0] || plb_info.in[1] ||
			plb_info.in[2] || plb_info.in[3]) {

			spin_lock_irqsave(&priv->irqmask_lock, irqflags);

			/* none of the interrupts have ocurred */
			if ((priv->out_vdc & plb_info.in[0]) ||
				(priv->out_sgx & plb_info.in[1]) ||
				(priv->out_sgx2 & plb_info.in[2]) ||
				(priv->out_mtx & plb_info.in[3])) {

				/* At least one of the interrupts has already occurred */
				plb_info.req_status = INT_STORED;

			} else {

				/* Wait for an interrupt to occur */
				priv->event_present = 0;
				spin_unlock_irqrestore(&priv->irqmask_lock, irqflags);

				DRM_WAIT_ON(ret, priv->event_queue, 20 * DRM_HZ,
					priv->event_present);

				if (ret) {
					plb_info.req_status = INT_TIMEOUT;
					break;
				}

				spin_lock_irqsave(&priv->irqmask_lock, irqflags);

				plb_info.req_status = INT_HANDLED;

			}

			plb_info.out[0] = priv->out_vdc;
			plb_info.out[1] = priv->out_sgx;
			plb_info.out[2] = priv->out_sgx2;
			plb_info.out[3] = priv->out_mtx;

			/* Clear the outstanding interrupts that have just been
			 * retrieved
			 */
			priv->out_vdc &= ~(plb_info.out[0] & plb_info.in[0]);
			priv->out_sgx &= ~(plb_info.out[1] & plb_info.in[1]) ;
			priv->out_sgx2 &= ~(plb_info.out[2] & plb_info.in[2]);
			priv->out_mtx &= ~(plb_info.out[3] & plb_info.in[3]);
			spin_unlock_irqrestore(&priv->irqmask_lock, irqflags);

		} else {

			/* Unsupported interrupt */
			plb_info.req_status = INT_NOOP;

		}

		break;

	case UNMASK_INT:

		spin_lock_irqsave(&priv->irqmask_lock, irqflags);
		PSB_WVDC32(0x00000000, IMR);
		spin_unlock_irqrestore(&priv->irqmask_lock, irqflags);

		break;

	case MASK_INT:

		spin_lock_irqsave(&priv->irqmask_lock, irqflags);
		PSB_WVDC32(0xFFFFFFFF, IMR);
		spin_unlock_irqrestore(&priv->irqmask_lock, irqflags);

		break;

	default:

		plb_info.req_status = INT_INVALID;

	}


	if (copy_to_user((void __user *) data, &plb_info, sizeof(plb_info))) {
		return -EFAULT;
	}

	return 0;
}
