/**
 * \ingroup liteqmux
 *
 * \file QmiService.h
 */
#ifndef __QMI_SERVICE_H__
#define __QMI_SERVICE_H__

#include <stdbool.h>
#include <pthread.h>
#include "QmiTransaction.h"

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \ingroup liteqmux
 * 
 * This funcion prototype represents the callback which is called by the QMI service 
 * when a QMI indication is received.
 * 
 * @param[in] qmiPacket indication's QMI packet (note: this does not include QMUX header).
 *
 * @param[in] qmiPacketSize Number of bytes in indication's QMI packet.
 *
 * @param[in] pIndicationCallbackContext User context supplied when QMI service was initialized.
*/
typedef void (*QMI_INDICATION_CALLBACK)(
    uint8_t* qmiPacket, 
    uint16_t qmiPacketSize, 
    void* pIndicationCallbackContext
    );

// Forward decls.
struct QmuxTransport;

#define QMI_TRANSACTION_POOL_SIZE 10 /**< Number of concurrent QMI transactions that may be outstanding per QMI service. */

/**
 * \ingroup liteqmux
 * 
 *  This structure abstracts a QMI service.
 *  It is responsible issuing transaction IDs.
 *  It tracks QMI requests and responses using transaction objects from
 *  a pool.
 *  It invokes a user callback when QMI indications which match 
 *  its client ID (or a broadcast ID) are received from the transport 
 *  layer. 
 *
 *  \param  pNext
 *          - A link so we can store this object in a linked list by the
 *            transport layer.
 * 
 *  \param  serviceType
 *          - QMI service type, CTL, NAS, etc.
 *
 *  \param  clientId
 *          - QMI client ID
 *
 *  \param  pTransport
 *          - Pointer to underlying QMUX transport object. 
 *
 *  \param  transactionId
 *          - QMI transaction ID. 
 *            Upper limit 0xffff for regular QMI services,
 *            0xff for CTL service.
 *             
 *  \param  transactionIdLock
 *          - Mutex to make transactionId access thread-safe. 
 *
 *  \param  transactionPool
 *          - Pool of transaction objects which may be assigned to 
 *            QMI requests made by clients of this service.
 *
 *  \param  transactionPoolLock
 *          - Mutex to make transactionPool access thread-safe
 *
 *  \param  pIndicationCallback
 *          - User callback for QMI indications of this service.
 *
 *  \param  pIndicationCallbackContext
 *          - User context for user callback.
 *
 */
typedef struct QmiService
{
    struct QmiService* pNext;  
    uint8_t serviceType;
    uint8_t clientId;
    struct QmuxTransport* pTransport;
    uint16_t transactionId;
    pthread_mutex_t transactionIdLock;
    QmiTransaction transactionPool[QMI_TRANSACTION_POOL_SIZE];
    pthread_mutex_t transactionPoolLock;
    QMI_INDICATION_CALLBACK pIndicationCallback;
    void* pIndicationCallbackContext;
} QmiService;

/**
 * \ingroup liteqmux
 * 
 * Initialize QMI service.
 * @param[in] pThis         The primary object of this call.
 * @param[in] serviceType   QMI service e.g. NAS, etc.
 * @param[in] clientId      Value previously obtained by invoking 
 *                          CtlService_GetClientId
 * @param[in] pTransport    Pointer to underlying transport layer which 
 *                          is responsible for sending and receiving 
 *                          QMUX packets
 * @param[in] pIndicationCallback   User function to be be called when QMI
 *                                  indications are received.    
 * @param[in] pIndicationCallbackContext   A user context for callback.    
 *       
 */
void QmiService_Initialize(
    QmiService* pThis, 
    uint8_t serviceType, 
    uint8_t clientId,
    struct QmuxTransport* pTransport,
    QMI_INDICATION_CALLBACK pIndicationCallback,
    void* pIndicationCallbackContext
    );

/**
 * \ingroup liteqmux
 * 
 * Shut down QMI service.
 * Brings service to an uninitiaized state.
 * @param[in] pThis      The primary object of this call.
 *       
 */
void QmiService_ShutDown(QmiService* pThis);

/**
 * \ingroup liteqmux
 * 
 * Returns next transaction ID for a QMI service.
 * @param[in] pThis      The primary object of this call.
 *       
 * @return transaction ID
*/
uint16_t QmiService_GetNextTransactionId(QmiService* pThis);

/**
 * \ingroup liteqmux
 * 
 * Send a QMI request to the device and return immediately.
 * 
 * @param[in] pThis         The primary object of this call.
 * @param[in] transactionId transaction ID obtained by previous call to
 *                          QmiService_GetNextTransactionId.
 * @param[in] qmiRequest    QMI request (this becomes the QMUX SDU).
 * @param[in] qmiRequestSize Size in bytes of QMI request.
 * @param[in] pResponseCallback Callback called when QMI request completes.
 * @param[in] pResponseCallbackContext Context for callback.
 *       
 * @return 0 on success, < 0 on failure. 
*/
int QmiService_SendRequest(
    QmiService* pThis,
    uint16_t transactionId,
    uint8_t* qmiRequest, 
    uint16_t qmiRequestSize, 
    QMI_RESPONSE_CALLBACK pResponseCallback, 
    void* pResponseCallbackContext);

/**
 * \ingroup liteqmux
 * 
 * Cancel an outstanding QMI request.
 * @param[in] pThis         The primary object of this call.
 * @param[in] transactionId Transaction ID of outstanding request.
 *       
 */
void QmiService_CancelTransaction(QmiService* pThis, uint16_t transactionId);

/**
 * \ingroup liteqmux
 * 
 * Process a received QMI packet.
 * This function is called by the QMUX transport layer.
 * 
 * @param[in] pThis         The primary object of this call.
 * @param[in] serviceType   QMI service type of received packet.
 * @param[in] clientId      QMI Client ID of received packet.
 * @param[in] qmiPacket     Pointer to QMI packet.
 * @param[in] qmiPacketSize Pointer to size in bytes of QMI packet.
 *       
 * @return true if packet is consumed by function, false if it should
 *         be offered to another QMI service 
*/
bool QmiService_HandleQmiPacket(
    QmiService* pThis,
    uint8_t serviceType, 
    uint8_t clientId, 
    uint8_t* qmiPacket, 
    uint16_t qmiPacketSize);


#ifdef __cplusplus
} /* extern "C" { */
#endif

#endif //__QMI_SERVICE_H__
